/* Entdecker&Erfinder
pH-Meter mit Linearer Kalibrierung (Steigung + Offset)
Optimierte Version für weite pH-Bereiche
Version: 2.0 mit Linearer Kalibrierung
*/

// ============================================
// 1. BIBLIOTHEKEN EINBINDEN
// ============================================
#include <Wire.h>
#include <LiquidCrystal_I2C.h>
#include <Keypad.h>

// ============================================
// 2. LCD-DISPLAY EINRICHTEN
// ============================================
LiquidCrystal_I2C lcd(0x27, 16, 2);

// ============================================
// 3. TASTENFELD (KEYPAD) EINRICHTEN
// ============================================
const byte ANZAHL_ZEILEN = 4;
const byte ANZAHL_SPALTEN = 4;

char tasteBelegung[ANZAHL_ZEILEN][ANZAHL_SPALTEN] = {
  {'1','2','3','A'},
  {'4','5','6','B'},
  {'7','8','9','C'},
  {'*','0','#','D'}
};

byte zeilenPins[ANZAHL_ZEILEN] = {9, 8, 7, 6};
byte spaltenPins[ANZAHL_SPALTEN] = {5, 4, 3, 2};

Keypad meinKeypad = Keypad(
  makeKeymap(tasteBelegung),
  zeilenPins,
  spaltenPins,
  ANZAHL_ZEILEN,
  ANZAHL_SPALTEN
);

// ============================================
// 4. PIN-DEFINITIONEN
// ============================================
const int statusLED = 13;

// ============================================
// 5. DATENSTRUKTUREN
// ============================================
struct KalibrierPunkt {
  float sollWert;    // Erwarteter pH-Wert
  float istSpannung; // Gemessene Spannung (mV)
  bool erledigt;
};

// ============================================
// 6. GLOBALE VARIABLEN - NEU MIT STEUERUNG
// ============================================
KalibrierPunkt kalibrierPunkte[3];
float kalibrierSteigung = 1.0;    // Steigungsfaktor (default = 1)
float kalibrierOffset = 0.0;      // Offset (pH-Einheiten)
float kalibrierSpannung0 = 0.0;   //  Spannung bei pH 0 (theoretisch)
int anzahlPunkte = 0;
int aktuellerPunkt = 0;
String eingabePuffer = "";
float aktuellerPH = 7.00;
float aktuellerRohWert = 0.0;     // Rohspannung in mV

// ============================================
// 7. PROGRAMMZUSTÄNDE
// ============================================
enum ProgrammZustand {
  ZUSTAND_NORMAL,
  ZUSTAND_KALIB_START,
  ZUSTAND_ANZAHL_EINGABE,
  ZUSTAND_SOLLWERT_EINGABE,
  ZUSTAND_MESSUNG,
  ZUSTAND_ERGEBNIS,
  ZUSTAND_INFO_ANZEIGE          //Info über Kalibrierung
};

ProgrammZustand aktuellerZustand = ZUSTAND_NORMAL;

// ============================================
// 8. MESSPUFFER
// ============================================
int messPuffer[10];
unsigned long mittelwert;

// ============================================
// 9. SETUP-FUNKTION
// ============================================
void setup() {
  pinMode(statusLED, OUTPUT);
  
  lcd.init();
  lcd.backlight();
  
  zeigeWillkommensbildschirm();
  zeigeEinfuehrung();
  resetKalibrierDaten();
  zeigeHauptbildschirm();
  
  // NEU: Standard-Steigung aus Nernst-Gleichung (59.16 mV/pH bei 25°C)
  // Unsere Elektrode: -5.70 * V + 28.55 entspricht ~ -57.0 mV/pH
  kalibrierSteigung = -57.0; // mV pro pH (negativ für pH-Elektroden)
}

// ============================================
// 10. LOOP-FUNKTION
// ============================================
void loop() {
  char gedrueckteTaste = meinKeypad.getKey();
  
  if (gedrueckteTaste) {
    digitalWrite(statusLED, HIGH);
    verarbeiteTaste(gedrueckteTaste);
    delay(50);
    digitalWrite(statusLED, LOW);
  }
  
  switch(aktuellerZustand) {
    case ZUSTAND_NORMAL:
      messeRegelmaessigPH();
      break;
      
    case ZUSTAND_MESSUNG:
      if (aktuellerPunkt < anzahlPunkte && 
          !kalibrierPunkte[aktuellerPunkt].erledigt) {
        messeKalibrierung();
        zeigeKalibrierMessung();
      }
      break;
      
    default:
      break;
  }
  
  delay(10);
}

// ============================================
// 11. TASTENVERARBEITUNG - ERWEITERT
// ============================================
void verarbeiteTaste(char taste) {
  lcd.setCursor(15, 0);
  lcd.print(taste);
  delay(80);
  lcd.setCursor(15, 0);
  lcd.print(" ");
  
  switch(taste) {
    case 'A':
      if (aktuellerZustand == ZUSTAND_NORMAL) {
        starteKalibrierung();
      } else if (aktuellerZustand == ZUSTAND_INFO_ANZEIGE) {
        // Zurück zum Normalmodus
        aktuellerZustand = ZUSTAND_NORMAL;
        zeigeHauptbildschirm();
      }
      break;
      
    case 'B':
      if (aktuellerZustand == ZUSTAND_MESSUNG) {
        beendeMessung();
      } else if (aktuellerZustand == ZUSTAND_NORMAL) {
        //  Info über aktuelle Kalibrierung anzeigen
        zeigeKalibrierInfo();
      }
      break;
      
    case 'C':
      lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print("Bestaetigt!");
      delay(500);
      bestaetigeEingabe();
      break;
      
    case 'D':
      if (aktuellerZustand != ZUSTAND_NORMAL && 
          aktuellerZustand != ZUSTAND_INFO_ANZEIGE) {
        abbrechenKalibrierung();
      }
      break;
      
    case '*':
      if ((aktuellerZustand == ZUSTAND_SOLLWERT_EINGABE || 
           aktuellerZustand == ZUSTAND_ANZAHL_EINGABE) && 
          !hatDezimalpunkt(eingabePuffer)) {
        eingabePuffer += '.';
        aktualisiereEingabeAnzeige();
      }
      break;
      
    case '#':
      if (aktuellerZustand == ZUSTAND_SOLLWERT_EINGABE || 
          aktuellerZustand == ZUSTAND_ANZAHL_EINGABE) {
        eingabePuffer = "";
        aktualisiereEingabeAnzeige();
      } else if (aktuellerZustand == ZUSTAND_NORMAL) {
        //  Offset auf 0 zurücksetzen
        kalibrierOffset = 0.0;
        kalibrierSteigung = -57.0; // Standardwert
        lcd.clear();
        lcd.setCursor(0, 0);
        lcd.print("Kalibrierung");
        lcd.setCursor(0, 1);
        lcd.print("zurueckgesetzt!");
        delay(2000);
        zeigeHauptbildschirm();
      }
      break;
      
    default:
      if (isdigit(taste)) {
        if (aktuellerZustand == ZUSTAND_SOLLWERT_EINGABE || 
            aktuellerZustand == ZUSTAND_ANZAHL_EINGABE) {
          if (eingabePuffer.length() < 6) {
            eingabePuffer += taste;
            aktualisiereEingabeAnzeige();
          }
        }
      }
      break;
  }
}

// ============================================
// 12. KALIBRIERUNGS-FUNKTIONEN
// ============================================
void starteKalibrierung() {
  aktuellerZustand = ZUSTAND_KALIB_START;
  anzahlPunkte = 0;
  aktuellerPunkt = 0;
  eingabePuffer = "";
  
  for (int i = 0; i < 3; i++) {
    kalibrierPunkte[i].erledigt = false;
  }
  
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Linear-Kalibrier.");
  lcd.setCursor(0, 1);
  lcd.print("C=Start  D=Abbruch");
  
  blinkLED(1, 500);
}

void bestaetigeEingabe() {
  switch(aktuellerZustand) {
    case ZUSTAND_KALIB_START:
      aktuellerZustand = ZUSTAND_ANZAHL_EINGABE;
      lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print("Wie viele Punkte?");
      lcd.setCursor(0, 1);
      lcd.print("(2-3): _   C=OK");
      break;
      
    case ZUSTAND_ANZAHL_EINGABE:
      if (eingabePuffer.length() > 0) {
        anzahlPunkte = eingabePuffer.toInt();
        // Mindestens 2 Punkte für lineare Kalibrierung!
        if (anzahlPunkte >= 2 && anzahlPunkte <= 3) {
          eingabePuffer = "";
          aktuellerPunkt = 0;
          zeigeSollwertEingabe();
        } else {
          fehlerAnzeige("Nur 2-3 Punkte!");
          eingabePuffer = "";
        }
      }
      break;
      
    case ZUSTAND_SOLLWERT_EINGABE:
      if (eingabePuffer.length() > 0) {
        float wert = eingabePuffer.toFloat();
        if (wert >= 0 && wert <= 14) {
          kalibrierPunkte[aktuellerPunkt].sollWert = wert;
          eingabePuffer = "";
          
          lcd.clear();
          lcd.setCursor(0, 0);
          lcd.print("Punkt ");
          lcd.print(aktuellerPunkt + 1);
          lcd.print(" von ");
          lcd.print(anzahlPunkte);
          lcd.setCursor(0, 1);
          lcd.print("Soll-pH: ");
          lcd.print(wert, 2);
          
          delay(1500);
          
          aktuellerZustand = ZUSTAND_MESSUNG;
          lcd.clear();
          lcd.setCursor(0, 0);
          lcd.print("Messpunkt ");
          lcd.print(aktuellerPunkt + 1);
          lcd.setCursor(0, 1);
          lcd.print("B=Messung starten");
          
          blinkLED(2, 200);
        } else {
          fehlerAnzeige("pH 0-14!");
          eingabePuffer = "";
        }
      }
      break;
      
    case ZUSTAND_ERGEBNIS:
      aktuellerZustand = ZUSTAND_NORMAL;
      
      lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print("Kalibrierung OK!");
      lcd.setCursor(0, 1);
      lcd.print("Steig: ");
      lcd.print(kalibrierSteigung, 1);
      lcd.print(" mV/pH");
      
      blinkLED(3, 200);
      delay(3000);
      
      zeigeKalibrierInfo();  //  Direkt Info anzeigen
      break;
  }
}

void beendeMessung() {
  if (aktuellerPunkt < anzahlPunkte) {
    // NEU: Spannung speichern statt pH-Wert!
    kalibrierPunkte[aktuellerPunkt].istSpannung = aktuellerRohWert;
    kalibrierPunkte[aktuellerPunkt].erledigt = true;
    
    lcd.clear();
    lcd.setCursor(0, 0);
    lcd.print("Punkt ");
    lcd.print(aktuellerPunkt + 1);
    lcd.print(" fertig!");
    lcd.setCursor(0, 1);
    lcd.print("Spann: ");
    lcd.print(kalibrierPunkte[aktuellerPunkt].istSpannung, 1);
    lcd.print(" mV");
    
    delay(2000);
    
    aktuellerPunkt++;
    if (aktuellerPunkt < anzahlPunkte) {
      zeigeSollwertEingabe();
    } else {
      // Alle Punkte fertig - Lineare Regression berechnen
      berechneLineareKalibrierung();
      aktuellerZustand = ZUSTAND_ERGEBNIS;
      zeigeKalibrierErgebnis();
    }
  }
}

void zeigeSollwertEingabe() {
  aktuellerZustand = ZUSTAND_SOLLWERT_EINGABE;
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Punkt ");
  lcd.print(aktuellerPunkt + 1);
  lcd.print(" Soll-pH?");
  lcd.setCursor(0, 1);
  lcd.print("Wert: _   C=OK");
}

// ============================================
// 13. pH-MESS-FUNKTIONEN - ÜBERARBEITET
// ============================================
void messeRegelmaessigPH() {
  static unsigned long letzteMessung = 0;
  
  if (millis() - letzteMessung >= 1000) {
    messePH();
    zeigeMesswert();
    letzteMessung = millis();
  }
}

void messePH() {
  // 1. Rohwerte aufnehmen
  for(int i = 0; i < 10; i++) {
    messPuffer[i] = analogRead(A0);
    delay(30);
  }
  
  // 2. Sortieren
  for(int i = 0; i < 9; i++) {
    for(int j = i+1; j < 10; j++) {
      if(messPuffer[i] > messPuffer[j]) {
        int temp = messPuffer[i];
        messPuffer[i] = messPuffer[j];
        messPuffer[j] = temp;
      }
    }
  }
  
  // 3. Mittelwert berechnen
  mittelwert = 0;
  for(int i = 2; i < 8; i++) {
    mittelwert += messPuffer[i];
  }
  
  // 4. NEU: Spannung in mV berechnen (0-5000mV bei 5V)
  aktuellerRohWert = (float)mittelwert * 5000.0 / 1024 / 6;
  
  // 5. pH-Wert mit linearer Kalibrierung berechnen
  // Formel: pH = (Spannung - OffsetSpannung) / Steigung
  // Umgestellt aus: Spannung = Steigung * pH + OffsetSpannung
  aktuellerPH = (aktuellerRohWert - kalibrierSpannung0) / kalibrierSteigung;
}

void messeKalibrierung() {
  // Für Kalibrierung nur Rohspannung messen
  for(int i = 0; i < 10; i++) {
    messPuffer[i] = analogRead(A0);
    delay(50);
  }
  
  // Sortieren
  for(int i = 0; i < 9; i++) {
    for(int j = i+1; j < 10; j++) {
      if(messPuffer[i] > messPuffer[j]) {
        int temp = messPuffer[i];
        messPuffer[i] = messPuffer[j];
        messPuffer[j] = temp;
      }
    }
  }
  
  // Mittelwert berechnen
  mittelwert = 0;
  for(int i = 2; i < 8; i++) {
    mittelwert += messPuffer[i];
  }
  
  // NEU: Nur Rohspannung berechnen (in mV)
  aktuellerRohWert = (float)mittelwert * 5000.0 / 1024 / 6;
}

// ============================================
// 14. BERECHNUNGS-FUNKTIONEN 
// ============================================
void berechneLineareKalibrierung() {
  /*
  Lineare Regression für 2 oder 3 Punkte:
  y = m*x + b
  wobei: y = Spannung (mV), x = pH-Wert
        m = Steigung (mV/pH)
        b = Offset-Spannung (Spannung bei pH 0)
  
  Dann: pH = (Spannung - b) / m
  */
  
  if (anzahlPunkte == 2) {
    // Einfache 2-Punkt-Kalibrierung
    int p1 = 0, p2 = 1;
    
    // Steigung m = ΔSpannung / ΔpH
    kalibrierSteigung = (kalibrierPunkte[p2].istSpannung - 
                        kalibrierPunkte[p1].istSpannung) /
                       (kalibrierPunkte[p2].sollWert - 
                        kalibrierPunkte[p1].sollWert);
    
    // Offset b = Spannung1 - m * pH1
    kalibrierSpannung0 = kalibrierPunkte[p1].istSpannung - 
                        (kalibrierSteigung * kalibrierPunkte[p1].sollWert);
    
  } else if (anzahlPunkte == 3) {
    // Lineare Regression mit 3 Punkten
    float sumX = 0, sumY = 0, sumXY = 0, sumX2 = 0;
    int n = 0;
    
    for (int i = 0; i < 3; i++) {
      if (kalibrierPunkte[i].erledigt) {
        float x = kalibrierPunkte[i].sollWert;
        float y = kalibrierPunkte[i].istSpannung;
        
        sumX += x;
        sumY += y;
        sumXY += x * y;
        sumX2 += x * x;
        n++;
      }
    }
    
    if (n > 1) {
      // Formeln für lineare Regression
      float m = (n * sumXY - sumX * sumY) / (n * sumX2 - sumX * sumX);
      float b = (sumY - m * sumX) / n;
      
      kalibrierSteigung = m;
      kalibrierSpannung0 = b;
    }
  }
  
  //  Offset in pH-Einheiten für Kompatibilität
  kalibrierOffset = -kalibrierSpannung0 / kalibrierSteigung;
}

// ============================================
// 15. ANZEIGE-FUNKTIONEN - ERWEITERT
// ============================================
void zeigeWillkommensbildschirm() {
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("pH-Meter v4.0");
  lcd.setCursor(0, 1);
  lcd.print("Linear-Kalibrier.");
  blinkLED(2, 300);
  delay(2500);
}

void zeigeEinfuehrung() {
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("2-Punkt-Kalib.");
  lcd.setCursor(0, 1);
  lcd.print("A=Start  B=Info");
  delay(2000);
}

void zeigeHauptbildschirm() {
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("pH: ");
  lcd.print(aktuellerPH, 2);
  lcd.setCursor(0, 1);
  lcd.print("A=Kalib  B=Info");
}

void zeigeMesswert() {
  // Zeigt auch Rohspannung an
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("pH:");
  lcd.print(aktuellerPH, 2);
  lcd.print(" Sp:");
  lcd.print((int)aktuellerRohWert);
  lcd.print("mV");
  
  lcd.setCursor(0, 1);
  lcd.print("A=Kalib  B=Info");
}

void zeigeKalibrierMessung() {
  // Zeigt Rohspannung während Kalibrierung
  lcd.setCursor(0, 0);
  lcd.print("Messung Punkt ");
  lcd.print(aktuellerPunkt + 1);
  lcd.setCursor(0, 1);
  lcd.print(aktuellerRohWert, 1);
  lcd.print(" mV   B=Stop");
}

void zeigeKalibrierErgebnis() {
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Kalibrier-Ergebnis");
  delay(1500);
  
  // Steigung anzeigen
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Steigung:");
  lcd.setCursor(0, 1);
  lcd.print(kalibrierSteigung, 1);
  lcd.print(" mV/pH  C=OK");
  delay(3000);
  
  // Offset anzeigen
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Offset:");
  lcd.setCursor(0, 1);
  lcd.print(kalibrierSpannung0, 0);
  lcd.print(" mV bei pH 0");
  delay(3000);
}

// NEUE FUNKTION: Kalibrier-Info anzeigen
void zeigeKalibrierInfo() {
  aktuellerZustand = ZUSTAND_INFO_ANZEIGE;
  
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Akt. Kalibrierung:");
  
  lcd.setCursor(0, 1);
  lcd.print("m=");
  lcd.print(kalibrierSteigung, 1);
  lcd.print(" b=");
  lcd.print(kalibrierSpannung0, 0);
  
  delay(3000);
  
  // Kalibrierpunkte anzeigen
  if (anzahlPunkte > 0) {
    for (int i = 0; i < anzahlPunkte; i++) {
      if (kalibrierPunkte[i].erledigt) {
        lcd.clear();
        lcd.setCursor(0, 0);
        lcd.print("P");
        lcd.print(i+1);
        lcd.print(": Soll ");
        lcd.print(kalibrierPunkte[i].sollWert, 2);
        
        lcd.setCursor(0, 1);
        lcd.print("Ist ");
        lcd.print(kalibrierPunkte[i].istSpannung, 0);
        lcd.print(" mV");
        
        delay(2000);
      }
    }
  }
  
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("A=Zurueck");
  lcd.setCursor(0, 1);
  lcd.print("B=Nochmal anzeigen");
}

// ============================================
// 16. HILFSFUNKTIONEN
// ============================================
void resetKalibrierDaten() {
  for (int i = 0; i < 3; i++) {
    kalibrierPunkte[i].erledigt = false;
    kalibrierPunkte[i].sollWert = 0;
    kalibrierPunkte[i].istSpannung = 0;
  }
}

void aktualisiereEingabeAnzeige() {
  lcd.setCursor(6, 1);
  lcd.print("      ");
  lcd.setCursor(6, 1);
  lcd.print(eingabePuffer);
}

bool hatDezimalpunkt(String text) {
  for (int i = 0; i < text.length(); i++) {
    if (text.charAt(i) == '.') return true;
  }
  return false;
}

void fehlerAnzeige(String nachricht) {
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Eingabefehler!");
  lcd.setCursor(0, 1);
  lcd.print(nachricht);
  
  blinkLED(5, 100);
  delay(2000);
  
  if (aktuellerZustand == ZUSTAND_ANZAHL_EINGABE) {
    lcd.clear();
    lcd.setCursor(0, 0);
    lcd.print("Wie viele Punkte?");
    lcd.setCursor(0, 1);
    lcd.print("(2-3): _   C=OK");
  } else if (aktuellerZustand == ZUSTAND_SOLLWERT_EINGABE) {
    zeigeSollwertEingabe();
  }
}

void abbrechenKalibrierung() {
  aktuellerZustand = ZUSTAND_NORMAL;
  
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Kalibrierung");
  lcd.setCursor(0, 1);
  lcd.print("abgebrochen");
  
  blinkLED(5, 100);
  delay(1500);
  zeigeHauptbildschirm();
}

void blinkLED(int anzahl, int dauer) {
  for (int i = 0; i < anzahl; i++) {
    digitalWrite(statusLED, HIGH);
    delay(dauer);
    digitalWrite(statusLED, LOW);
    if (i < anzahl - 1) delay(dauer);
  }
}