/*
pH-Meter mit Kalibrierfunktion
Einfache Version für Arduino-Einsteiger
Autor: Dein Name
Version: 1.0 für Einsteiger
*/

// ============================================
// 1. BIBLIOTHEKEN EINBINDEN
// ============================================
/*
Bibliotheken sind vorgefertigte Code-Sammlungen.
Wir laden sie, um Hardware wie LCD und Keypad einfach ansteuern zu können.
*/
#include <Wire.h>              // Für I2C-Kommunikation (LCD)
#include <LiquidCrystal_I2C.h> // Für das LCD-Display
#include <Keypad.h>            // Für das Tastenfeld

// ============================================
// 2. LCD-DISPLAY EINRICHTEN
// ============================================
/*
Wir erstellen ein LCD-Objekt mit Adresse 0x27,
16 Zeichen pro Zeile und 2 Zeilen.
*/
LiquidCrystal_I2C lcd(0x27, 16, 2);  // Adresse, Spalten, Zeilen

// ============================================
// 3. TASTENFELD (KEYPAD) EINRICHTEN
// ============================================
/*
Unser Keypad hat 4 Zeilen und 4 Spalten.
Wir definieren zuerst, welche Tasten wo sind.
*/
const byte ANZAHL_ZEILEN = 4;      // 4 Zeilen
const byte ANZAHL_SPALTEN = 4;     // 4 Spalten

// Tastenbelegung - so sind die Tasten angeordnet
char tasteBelegung[ANZAHL_ZEILEN][ANZAHL_SPALTEN] = {
  {'1','2','3','A'},  // Erste Zeile
  {'4','5','6','B'},  // Zweite Zeile
  {'7','8','9','C'},  // Dritte Zeile
  {'*','0','#','D'}   // Vierte Zeile
};

// An welche Arduino-Pins die Keypad-Zeilen angeschlossen sind
byte zeilenPins[ANZAHL_ZEILEN] = {9, 8, 7, 6};

// An welche Arduino-Pins die Keypad-Spalten angeschlossen sind
byte spaltenPins[ANZAHL_SPALTEN] = {5, 4, 3, 2};

// Keypad-Objekt erstellen
Keypad meinKeypad = Keypad(
  makeKeymap(tasteBelegung),  // Tastenbelegung
  zeilenPins,                 // Zeilen-Pins
  spaltenPins,                // Spalten-Pins
  ANZAHL_ZEILEN,              // Anzahl Zeilen
  ANZAHL_SPALTEN              // Anzahl Spalten
);

// ============================================
// 4. PIN-DEFINITIONEN
// ============================================
const int statusLED = 13;  // Status-LED an Pin 13

// ============================================
// 5. DATENSTRUKTUREN (Wie wir Daten speichern)
// ============================================
/*
Eine "Struktur" ist wie ein Formular mit mehreren Feldern.
Hier speichern wir für jeden Kalibrierpunkt:
- Sollwert: Welchen pH-Wert wir erwarten
- Istwert: Was wir wirklich messen
- Erledigt: Ob dieser Punkt schon gemessen wurde
*/
struct KalibrierPunkt {
  float sollWert;    // Erwarteter pH-Wert (z.B. 7.00)
  float istWert;     // Gemessener pH-Wert
  bool erledigt;     // True = fertig gemessen
};

// ============================================
// 6. GLOBALE VARIABLEN
// ============================================
/*
Globale Variablen sind im ganzen Programm sichtbar.
Sie behalten ihren Wert zwischen den loop()-Durchläufen.
*/
KalibrierPunkt kalibrierPunkte[3];  // Array für bis zu 3 Punkte
float kalibrierOffset = 0.0;        // Korrekturwert nach Kalibrierung
float aktuellerPH = 7.00;           // Aktuell gemessener pH-Wert
int anzahlPunkte = 0;               // Wie viele Punkte kalibriert werden
int aktuellerPunkt = 0;             // Welcher Punkt gerade bearbeitet wird
String eingabePuffer = "";          // Speichert Tastatureingaben

// ============================================
// 7. PROGRAMMZUSTÄNDE (State Machine)
// ============================================
/*
Statt vieler if-Abfragen verwenden wir Zustände.
Das Programm kann nur in einem Zustand gleichzeitig sein.
*/
enum ProgrammZustand {
  ZUSTAND_NORMAL,          // Normale pH-Messung
  ZUSTAND_KALIB_START,     // Kalibrierung starten
  ZUSTAND_ANZAHL_EINGABE,  // Anzahl Punkte eingeben
  ZUSTAND_SOLLWERT_EINGABE,// Soll-pH-Wert eingeben
  ZUSTAND_MESSUNG,         // Messung läuft
  ZUSTAND_ERGEBNIS         // Kalibrierergebnis anzeigen
};

ProgrammZustand aktuellerZustand = ZUSTAND_NORMAL;  // Startzustand

// ============================================
// 8. MESSPUFFER FÜR pH-WERTE
// ============================================
/*
Wir messen 10 Werte, sortieren sie und nehmen den Mittelwert
von den 6 mittleren Werten. Das reduziert Messrauschen.
*/
int messPuffer[10];        // Speicher für 10 Messungen
unsigned long mittelwert;  // Berechneter Mittelwert

// ============================================
// 9. SETUP-FUNKTION (Wird einmal beim Start ausgeführt)
// ============================================
void setup() {
  // Status-LED als Ausgang definieren
  pinMode(statusLED, OUTPUT);
  
  // LCD initialisieren
  lcd.init();           // Display starten
  lcd.backlight();      // Hintergrundbeleuchtung einschalten
  
  // Willkommensbildschirm anzeigen
  zeigeWillkommensbildschirm();
  
  // Kurze Einführung anzeigen
  zeigeEinfuehrung();
  
  // Kalibrierungsdaten zurücksetzen
  resetKalibrierDaten();
  
  // Hauptbildschirm anzeigen
  zeigeHauptbildschirm();
}

// ============================================
// 10. LOOP-FUNKTION (Wird immer wieder ausgeführt)
// ============================================
void loop() {
  // 1. Prüfen, ob eine Taste gedrückt wurde
  char gedrueckteTaste = meinKeypad.getKey();
  
  if (gedrueckteTaste) {
    // Taste wurde gedrückt
    digitalWrite(statusLED, HIGH);  // LED einschalten
    verarbeiteTaste(gedrueckteTaste);  // Taste verarbeiten
    delay(50);  // Kurze Pause für Entprellung
    digitalWrite(statusLED, LOW);   // LED ausschalten
  }
  
  // 2. Je nach Zustand unterschiedliche Aktionen
  switch(aktuellerZustand) {
    case ZUSTAND_NORMAL:
      // Im Normalmodus regelmäßig pH messen
      messeRegelmaessigPH();
      break;
      
    case ZUSTAND_MESSUNG:
      // Im Messmodus Kalibrierung durchführen
      if (aktuellerPunkt < anzahlPunkte && 
          !kalibrierPunkte[aktuellerPunkt].erledigt) {
        messeKalibrierung();
        zeigeKalibrierMessung();
      }
      break;
      
    // In anderen Zuständen passiert im Loop nichts Besonderes
    default:
      break;
  }
  
  delay(10);  // Kurze Pause, um den Prozessor nicht zu überlasten
}

// ============================================
// 11. TASTENVERARBEITUNG
// ============================================
void verarbeiteTaste(char taste) {
  // Kurze Anzeige der gedrückten Taste (rechts oben)
  lcd.setCursor(15, 0);
  lcd.print(taste);
  delay(80);
  lcd.setCursor(15, 0);
  lcd.print(" ");
  
  // Je nach Taste unterschiedliche Aktionen
  switch(taste) {
    case 'A':
      // Taste A: Kalibrierung starten (nur im Normalmodus)
      if (aktuellerZustand == ZUSTAND_NORMAL) {
        starteKalibrierung();
      }
      break;
      
    case 'B':
      // Taste B: Messung beenden (nur im Messmodus)
      if (aktuellerZustand == ZUSTAND_MESSUNG) {
        beendeMessung();
      }
      break;
      
    case 'C':
      // Taste C: BESTÄTIGEN
      lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print("Bestaetigt!");
      delay(500);
      bestaetigeEingabe();
      break;
      
    case 'D':
      // Taste D: ABBRECHEN (wenn nicht im Normalmodus)
      if (aktuellerZustand != ZUSTAND_NORMAL) {
        abbrechenKalibrierung();
      }
      break;
      
    case '*':
      // Stern: Dezimalpunkt eingeben
      if ((aktuellerZustand == ZUSTAND_SOLLWERT_EINGABE || 
           aktuellerZustand == ZUSTAND_ANZAHL_EINGABE) && 
          !hatDezimalpunkt(eingabePuffer)) {
        eingabePuffer += '.';
        aktualisiereEingabeAnzeige();
      }
      break;
      
    case '#':
      // Raute: Eingabe löschen
      if (aktuellerZustand == ZUSTAND_SOLLWERT_EINGABE || 
          aktuellerZustand == ZUSTAND_ANZAHL_EINGABE) {
        eingabePuffer = "";
        aktualisiereEingabeAnzeige();
      }
      break;
      
    default:
      // Zifferntasten 0-9
      if (isdigit(taste)) {  // Prüft, ob es eine Ziffer ist
        if (aktuellerZustand == ZUSTAND_SOLLWERT_EINGABE || 
            aktuellerZustand == ZUSTAND_ANZAHL_EINGABE) {
          if (eingabePuffer.length() < 6) {  // Max. 6 Zeichen
            eingabePuffer += taste;
            aktualisiereEingabeAnzeige();
          }
        }
      }
      break;
  }
}

// ============================================
// 12. KALIBRIERUNGS-FUNKTIONEN
// ============================================

void starteKalibrierung() {
  // Kalibrierung vorbereiten
  aktuellerZustand = ZUSTAND_KALIB_START;
  anzahlPunkte = 0;
  aktuellerPunkt = 0;
  eingabePuffer = "";
  
  // Alle Punkte zurücksetzen
  for (int i = 0; i < 3; i++) {
    kalibrierPunkte[i].erledigt = false;
  }
  
  // Bildschirm für Kalibrierungsstart
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Mehrpunkt-Kalib.");
  lcd.setCursor(0, 1);
  lcd.print("C=Start  D=Abbruch");
  
  blinkLED(1, 500);  // 1x blinken
}

void bestaetigeEingabe() {
  // Was passiert bei "Bestätigen" hängt vom aktuellen Zustand ab
  switch(aktuellerZustand) {
    case ZUSTAND_KALIB_START:
      // Vom Start zur Anzahl-Eingabe wechseln
      aktuellerZustand = ZUSTAND_ANZAHL_EINGABE;
      lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print("Wie viele Punkte?");
      lcd.setCursor(0, 1);
      lcd.print("(1-3): _   C=OK");
      break;
      
    case ZUSTAND_ANZAHL_EINGABE:
      // Anzahl der Punkte verarbeiten
      if (eingabePuffer.length() > 0) {
        anzahlPunkte = eingabePuffer.toInt();  // String zu Zahl
        if (anzahlPunkte >= 1 && anzahlPunkte <= 3) {
          eingabePuffer = "";
          aktuellerPunkt = 0;
          zeigeSollwertEingabe();
        } else {
          fehlerAnzeige("Nur 1-3 Punkte!");
          eingabePuffer = "";
        }
      }
      break;
      
    case ZUSTAND_SOLLWERT_EINGABE:
      // Sollwert für aktuellen Punkt speichern
      if (eingabePuffer.length() > 0) {
        float wert = eingabePuffer.toFloat();  // String zu Dezimalzahl
        if (wert >= 0 && wert <= 14) {  // pH-Werte sind 0-14
          kalibrierPunkte[aktuellerPunkt].sollWert = wert;
          eingabePuffer = "";
          
          // Bestätigung anzeigen
          lcd.clear();
          lcd.setCursor(0, 0);
          lcd.print("Punkt ");
          lcd.print(aktuellerPunkt + 1);
          lcd.print(" von ");
          lcd.print(anzahlPunkte);
          lcd.setCursor(0, 1);
          lcd.print("Soll-pH: ");
          lcd.print(wert, 2);  // 2 Nachkommastellen
          
          delay(1500);
          
          // Zum Messmodus wechseln
          aktuellerZustand = ZUSTAND_MESSUNG;
          lcd.clear();
          lcd.setCursor(0, 0);
          lcd.print("Messpunkt ");
          lcd.print(aktuellerPunkt + 1);
          lcd.setCursor(0, 1);
          lcd.print("B=Messung starten");
          
          blinkLED(2, 200);
        } else {
          fehlerAnzeige("pH 0-14!");
          eingabePuffer = "";
        }
      }
      break;
      
    case ZUSTAND_ERGEBNIS:
      // Zurück zum Normalmodus
      aktuellerZustand = ZUSTAND_NORMAL;
      
      // Erfolgsmeldung anzeigen
      lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print("Kalibrierung OK!");
      lcd.setCursor(0, 1);
      lcd.print("Offset: ");
      lcd.print(kalibrierOffset, 2);
      lcd.print(" pH");
      
      blinkLED(3, 200);
      delay(2500);
      
      zeigeHauptbildschirm();
      break;
  }
}

void beendeMessung() {
  // Messung für aktuellen Punkt abschließen
  if (aktuellerPunkt < anzahlPunkte) {
    // Gemessenen pH-Wert speichern
    kalibrierPunkte[aktuellerPunkt].istWert = aktuellerPH;
    kalibrierPunkte[aktuellerPunkt].erledigt = true;
    
    // Erfolgsmeldung
    lcd.clear();
    lcd.setCursor(0, 0);
    lcd.print("Punkt ");
    lcd.print(aktuellerPunkt + 1);
    lcd.print(" fertig!");
    lcd.setCursor(0, 1);
    lcd.print("C=naechster Punkt");
    
    delay(2000);
    
    // Zum nächsten Punkt oder zum Ergebnis
    aktuellerPunkt++;
    if (aktuellerPunkt < anzahlPunkte) {
      zeigeSollwertEingabe();
    } else {
      // Alle Punkte fertig - Ergebnis berechnen
      berechneKalibrierung();
      aktuellerZustand = ZUSTAND_ERGEBNIS;
      zeigeKalibrierErgebnis();
    }
  }
}

void zeigeSollwertEingabe() {
  // Eingabebildschirm für Soll-pH-Wert
  aktuellerZustand = ZUSTAND_SOLLWERT_EINGABE;
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Punkt ");
  lcd.print(aktuellerPunkt + 1);
  lcd.print(" Soll-pH?");
  lcd.setCursor(0, 1);
  lcd.print("Wert: _   C=OK");
}

// ============================================
// 13. pH-MESS-FUNKTIONEN
// ============================================

void messeRegelmaessigPH() {
  // pH-Wert jede Sekunde neu messen
  static unsigned long letzteMessung = 0;
  
  if (millis() - letzteMessung >= 1000) {  // 1000ms = 1 Sekunde
    messePH();
    zeigeMesswert();
    letzteMessung = millis();
  }
}

void messePH() {
  /*
  pH-Messung mit Rauschunterdrückung:
  1. 10 Messwerte aufnehmen
  2. Werte sortieren (aufsteigend)
  3. Die 4 extremen Werte wegwerfen
  4. Von den 6 mittleren Werten den Mittelwert nehmen
  */
  
  // 1. 10 Messwerte aufnehmen
  for(int i = 0; i < 10; i++) {
    messPuffer[i] = analogRead(A0);  // Lies Wert von pH-Sensor
    delay(30);  // Kurze Pause zwischen Messungen
  }
  
  // 2. Werte sortieren (Bubble-Sort, einfach aber langsam)
  for(int i = 0; i < 9; i++) {
    for(int j = i+1; j < 10; j++) {
      if(messPuffer[i] > messPuffer[j]) {
        // Werte tauschen
        int temp = messPuffer[i];
        messPuffer[i] = messPuffer[j];
        messPuffer[j] = temp;
      }
    }
  }
  
  // 3. Mittelwert der mittleren 6 Werte berechnen
  mittelwert = 0;
  for(int i = 2; i < 8; i++) {  // Index 2-7 (6 Werte)
    mittelwert += messPuffer[i];
  }
  
  // 4. Rohwert in pH umrechnen
  // Formel: pH = -5.70 * Spannung + 28.55 + Offset
  float spannung = (float)mittelwert * 5.0 / 1024 / 6;
  aktuellerPH = -5.70 * spannung + 28.55 + kalibrierOffset;
}

void messeKalibrierung() {
  // Ähnlich wie messePH(), aber ohne Offset
  for(int i = 0; i < 10; i++) {
    messPuffer[i] = analogRead(A0);
    delay(50);  // Etwas längere Pause für stabilere Werte
  }
  
  // Sortieren
  for(int i = 0; i < 9; i++) {
    for(int j = i+1; j < 10; j++) {
      if(messPuffer[i] > messPuffer[j]) {
        int temp = messPuffer[i];
        messPuffer[i] = messPuffer[j];
        messPuffer[j] = temp;
      }
    }
  }
  
  // Mittelwert berechnen
  mittelwert = 0;
  for(int i = 2; i < 8; i++) {
    mittelwert += messPuffer[i];
  }
  
  // Rohwert in pH umrechnen (OHNE Offset für Kalibrierung)
  float spannung = (float)mittelwert * 5.0 / 1024 / 6;
  aktuellerPH = -5.70 * spannung + 28.55;
}

// ============================================
// 14. BERECHNUNGS-FUNKTIONEN
// ============================================

void berechneKalibrierung() {
  /*
  Berechnet den Korrekturwert (Offset):
  Offset = Durchschnitt aller (Sollwert - Istwert)
  */
  float summeDifferenz = 0;
  int anzahl = 0;
  
  // Für jeden kalibrierten Punkt
  for (int i = 0; i < anzahlPunkte; i++) {
    if (kalibrierPunkte[i].erledigt) {
      float differenz = kalibrierPunkte[i].sollWert - kalibrierPunkte[i].istWert;
      summeDifferenz += differenz;
      anzahl++;
    }
  }
  
  // Durchschnitt berechnen
  if (anzahl > 0) {
    kalibrierOffset = summeDifferenz / anzahl;
  }
}

// ============================================
// 15. ANZEIGE-FUNKTIONEN
// ============================================

void zeigeWillkommensbildschirm() {
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("  pH-Meter v3.0 ");
  lcd.setCursor(0, 1);
  lcd.print("Taste C=best&tigen");
  blinkLED(2, 300);
  delay(2500);
}

void zeigeEinfuehrung() {
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Kalibrierung:");
  lcd.setCursor(0, 1);
  lcd.print("Taste A druecken");
  delay(2000);
}

void zeigeHauptbildschirm() {
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Aktueller pH:");
  lcd.setCursor(0, 1);
  lcd.print(aktuellerPH, 2);  // 2 Nachkommastellen
  lcd.print("   A=Menue");
}

void zeigeMesswert() {
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("pH: ");
  lcd.print(aktuellerPH, 2);
  
  lcd.setCursor(0, 1);
  if (kalibrierOffset == 0) {
    lcd.print("A=Kalibrieren");
  } else {
    lcd.print("A=Kalib   #=Offset");
  }
}

void zeigeKalibrierMessung() {
  // Zeigt aktuellen pH-Wert während der Kalibrierung
  lcd.setCursor(0, 1);
  lcd.print("pH: ");
  lcd.print(aktuellerPH, 2);
  lcd.print("  B=Stop");
}

void zeigeKalibrierErgebnis() {
  // Zusammenfassung der Kalibrierung
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Kalibrierung");
  lcd.setCursor(0, 1);
  lcd.print("abgeschlossen!");
  delay(2000);
  
  // Offset anzeigen
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Korrekturwert:");
  lcd.setCursor(0, 1);
  lcd.print(kalibrierOffset, 2);
  lcd.print(" pH  C=OK");
  blinkLED(1, 300);
}

// ============================================
// 16. HILFSFUNKTIONEN
// ============================================

void resetKalibrierDaten() {
  // Setzt alle Kalibrierdaten zurück
  for (int i = 0; i < 3; i++) {
    kalibrierPunkte[i].erledigt = false;
    kalibrierPunkte[i].sollWert = 0;
    kalibrierPunkte[i].istWert = 0;
  }
}

void aktualisiereEingabeAnzeige() {
  // Aktualisiert die Eingabeanzeige auf dem LCD
  lcd.setCursor(6, 1);
  lcd.print("      ");  // Alten Inhalt löschen
  lcd.setCursor(6, 1);
  lcd.print(eingabePuffer);  // Neuen Inhalt anzeigen
}

bool hatDezimalpunkt(String text) {
  // Prüft, ob ein String schon einen Dezimalpunkt enthält
  for (int i = 0; i < text.length(); i++) {
    if (text.charAt(i) == '.') return true;
  }
  return false;
}

void fehlerAnzeige(String nachricht) {
  // Zeigt eine Fehlermeldung an
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Eingabefehler!");
  lcd.setCursor(0, 1);
  lcd.print(nachricht);
  
  blinkLED(5, 100);  // Schnell blinken bei Fehler
  delay(2000);
  
  // Zurück zum vorherigen Bildschirm
  if (aktuellerZustand == ZUSTAND_ANZAHL_EINGABE) {
    lcd.clear();
    lcd.setCursor(0, 0);
    lcd.print("Wie viele Punkte?");
    lcd.setCursor(0, 1);
    lcd.print("(1-3): _   C=OK");
  } else if (aktuellerZustand == ZUSTAND_SOLLWERT_EINGABE) {
    zeigeSollwertEingabe();
  }
}

void abbrechenKalibrierung() {
  // Bricht die Kalibrierung ab
  aktuellerZustand = ZUSTAND_NORMAL;
  
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Kalibrierung");
  lcd.setCursor(0, 1);
  lcd.print("abgebrochen");
  
  blinkLED(5, 100);
  delay(1500);
  zeigeHauptbildschirm();
}

void blinkLED(int anzahl, int dauer) {
  // Lässt die Status-LED blinken
  for (int i = 0; i < anzahl; i++) {
    digitalWrite(statusLED, HIGH);
    delay(dauer);
    digitalWrite(statusLED, LOW);
    if (i < anzahl - 1) delay(dauer);
  }
}