#include <LiquidCrystal.h>

// LCD Pin-Belegung (16x2 Display)
LiquidCrystal lcd(7, 8, 9, 10, 11, 12);

// Angelehnt an das Beispiel von Electronoobs: https://electronoobs.com/eng_arduino_tut182.php
const int MQ_PIN = A0;           // Analogeingang für den MQ-3 Sensor
float RL_WERT = 18.7;            // RL Widerstand in Kilo-Ohm 
float R0 = 0;                    // R0 Widerstand in Kilo-Ohm (wird kalibriert)

// Kalibrierungsvariablen
float sensor_spannung = 0;
float RS = 0;
float sensor_wert = 0;
bool kalibriert = false;         // Status ob Kalibrierung durchgeführt wurde

// Messvariablen
const int LESEN_INTERVALL = 100;     // Zeit zwischen Messungen (ms)
const int LESEN_ANZAHL = 5;          // Anzahl der Messwerte für Mittelwert

// Zwei Punkte aus der doppel logarithmischen Auftragung aus MQ-3 Datenblatt
const float KALIB_X0 = 50;       
const float KALIB_Y0 = 0.18;    
const float KALIB_X1 = 500;     
const float KALIB_Y1 = 0.012;   

// Berechnung der Kurvenparameter
const float punkt0[] = { log10(KALIB_X0), log10(KALIB_Y0) };
const float punkt1[] = { log10(KALIB_X1), log10(KALIB_Y1) };
const float steigung = (punkt1[1] - punkt0[1]) / (punkt1[0] - punkt0[0]);
const float achsenabschnitt = punkt0[1] - punkt0[0] * steigung;

// Umrechnungskonstanten für Alkohol
const float MOLARE_MASSE_ETHANOL = 46.07;        // g/mol
const float LUFT_MOLARES_VOLUMEN = 24.5;         // l/mol bei Raumtemperatur (25°C)
const float PPM_TO_G_PER_L_FACTOR = (MOLARE_MASSE_ETHANOL / (LUFT_MOLARES_VOLUMEN * 1000.0));

// Display Modus - nur noch 3 Modi
enum DisplayModus {
  MODUS_KONZENTRATION,
  MODUS_GRAMM_PRO_LITER,
  MODUS_SPANNUNG
};
DisplayModus aktiver_modus = MODUS_KONZENTRATION;
unsigned long letzter_modus_wechsel = 0;
const unsigned long MODUS_WECHSEL_INTERVALL = 3000; // 3 Sekunden

void setup() {
  Serial.begin(9600);
  
  // LCD Display initialisieren
  lcd.begin(16, 2);
  lcd.print("MQ-3 Alkoholsensor");
  lcd.setCursor(0, 1);
  lcd.print("  Kalibrierung...");
  
  Serial.println("MQ-3 Alkoholsensor mit LCD");
  Serial.println("==========================");
  delay(2000);
  
  // Kalibrierung durchführen
  kalibriereSensor();
  kalibriert = true;
  
  lcd.clear();
  lcd.print("Kalibrierung OK");
  lcd.setCursor(0, 1);
  lcd.print("R0: ");
  lcd.print(R0, 3);
  lcd.print(" kΩ");
  Serial.print("Kalibrierung abgeschlossen. R0 = ");
  Serial.print(R0, 5);
  Serial.println(" kΩ");
  delay(2000);
}

void loop() {
  // Automatischer Moduswechsel alle 3 Sekunden
  if (millis() - letzter_modus_wechsel > MODUS_WECHSEL_INTERVALL) {
    wechsleDisplayModus();
    letzter_modus_wechsel = millis();
  }
  
  // Messung durchführen - Array auf 3 Elemente
  float messwerte[3]; 
  fuehreMessungDurch(messwerte);
  
  // Display aktualisieren basierend auf aktuellem Modus
  aktualisiereDisplay(messwerte);
  
  delay(500); // Kurze Pause zwischen Updates
}

void kalibriereSensor() {
  Serial.println("=== KALIBRIERUNG ===");
  sensor_wert = 0;
  
  // 100 Messwerte für genaue Kalibrierung
  for (int i = 0; i < 100; i++) {
    sensor_wert = sensor_wert + analogRead(MQ_PIN);
    delay(10);
  }
  
  // Durchschnitt berechnen
  sensor_wert = sensor_wert / 100.0;
  sensor_spannung = sensor_wert / 1024.0 * 5.0;
  
  Serial.print("Spannung: ");
  Serial.print(sensor_spannung, 4);
  Serial.println(" V");
  
  // RS und R0 berechnen (für reine Luft ohne Alkohol)
  RS = (5.0 - sensor_spannung) / sensor_spannung * RL_WERT;
  R0 = RS / 57.0;  
  
  Serial.print("RS: ");
  Serial.print(RS, 2);
  Serial.print(" kΩ, R0: ");
  Serial.print(R0, 5);
  Serial.println(" kΩ");
  Serial.println("====================");
  
  sensor_wert = 0;  // Reset für nächste Messung
}

void wechsleDisplayModus() {
  switch(aktiver_modus) {
    case MODUS_KONZENTRATION:
      aktiver_modus = MODUS_GRAMM_PRO_LITER;
      break;
    case MODUS_GRAMM_PRO_LITER:
      aktiver_modus = MODUS_SPANNUNG;
      break;
    case MODUS_SPANNUNG:
      aktiver_modus = MODUS_KONZENTRATION;
      break;
  }
}

void fuehreMessungDurch(float* ergebnisse) {
  // RS Widerstand messen (gemittelt)
  float rs_mittel = leseMQWiderstand(MQ_PIN);
  
  // Konzentration in ppm berechnen
  float konzentration_ppm = berechneKonzentration(rs_mittel / R0);
  
  // Alkoholkonzentration in g/l berechnen
  float alkohol_g_per_l = berechneGrammProLiter(konzentration_ppm);
  
  // Aktuelle Spannung messen
  float aktuelle_spannung = analogRead(MQ_PIN) / 1024.0 * 5.0;
  
  ergebnisse[0] = konzentration_ppm;
  ergebnisse[1] = alkohol_g_per_l;
  ergebnisse[2] = aktuelle_spannung;
  
  // Serial Output für Debugging
  Serial.print("RS: ");
  Serial.print(rs_mittel, 2);
  Serial.print(" kΩ | Konz: ");
  Serial.print(konzentration_ppm, 0);
  Serial.print(" ppm | g/l: ");
  Serial.print(alkohol_g_per_l, 6);  // 6 Dezimalstellen für kleine Werte
  Serial.print(" | Spannung: ");
  Serial.print(aktuelle_spannung, 3);
  Serial.println(" V");
}

// Berechnung der Alkoholkonzentration in g/l
float berechneGrammProLiter(float konzentration_ppm) {
  // Umrechnung: ppm → g/l
  // Formel: (ppm) * (molare Masse) / (molares Volumen * 1000)
  return konzentration_ppm * PPM_TO_G_PER_L_FACTOR;
}

void aktualisiereDisplay(float* werte) {
  lcd.clear();
  
  // Obere Zeile: Modusanzeige
  lcd.setCursor(0, 0);
  
  // Modusanzeige
  switch(aktiver_modus) {
    case MODUS_KONZENTRATION:
      lcd.print("Alkohol (ppm):");
      break;
    case MODUS_GRAMM_PRO_LITER:
      lcd.print("Alkohol (g/l):");
      break;
    case MODUS_SPANNUNG:
      lcd.print("Spannung (V):");
      break;
  }
  
  // Untere Zeile: Messwert
  lcd.setCursor(0, 1);
  switch(aktiver_modus) {
    case MODUS_KONZENTRATION:
      if (werte[0] < 1000) {
        lcd.print(werte[0], 0);
      } else {
        lcd.print(werte[0] / 1000, 2);
        lcd.print("k");
      }
      lcd.print(" ppm");
      break;
      
    case MODUS_GRAMM_PRO_LITER:
      lcd.print(werte[1], 6);  // 6 Dezimalstellen für kleine Werte
      lcd.print(" g/l");
      break;
      
    case MODUS_SPANNUNG:
      lcd.print(werte[2], 3);
      lcd.print(" V");
      break;
  }
}

// Liest den MQ-Widerstand mit Mehrfachmessung
float leseMQWiderstand(int mq_pin) {
  float rs = 0;
  for (int i = 0; i < LESEN_ANZAHL; i++) {
    rs += berechneWiderstand(analogRead(mq_pin));
    delay(LESEN_INTERVALL);
  }
  return rs / LESEN_ANZAHL;
}

// Berechnet Widerstand aus analogem Wert
float berechneWiderstand(int roh_wert) {
  return ((float)RL_WERT * (1023 - roh_wert) / roh_wert);
}

// Berechnet Konzentration aus RS/R0 Verhältnis
float berechneKonzentration(float rs_ro_verhaeltnis) {
  // Schutz vor ungültigen Werten
  if (rs_ro_verhaeltnis <= 0) {
    return 0;
  }
  return pow(10, achsenabschnitt + steigung * log10(rs_ro_verhaeltnis));
}